/*
 * Decompiled with CFR 0.152.
 */
package ch.systemsx.cisd.hdf5.h5ar;

import ch.systemsx.cisd.base.exceptions.IErrorStrategy;
import ch.systemsx.cisd.base.exceptions.IOExceptionUnchecked;
import ch.systemsx.cisd.base.unix.FileLinkType;
import ch.systemsx.cisd.base.unix.Unix;
import ch.systemsx.cisd.hdf5.IHDF5Reader;
import ch.systemsx.cisd.hdf5.h5ar.ArchiveEntry;
import ch.systemsx.cisd.hdf5.h5ar.ArchiverException;
import ch.systemsx.cisd.hdf5.h5ar.ArchivingStrategy;
import ch.systemsx.cisd.hdf5.h5ar.GroupCache;
import ch.systemsx.cisd.hdf5.h5ar.IArchiveEntryProcessor;
import ch.systemsx.cisd.hdf5.h5ar.IArchiveEntryVisitor;
import ch.systemsx.cisd.hdf5.h5ar.IdCache;
import ch.systemsx.cisd.hdf5.h5ar.LinkRecord;
import ch.systemsx.cisd.hdf5.h5ar.UnarchivingException;
import ch.systemsx.cisd.hdf5.h5ar.Utils;
import hdf.hdf5lib.exceptions.HDF5Exception;
import hdf.hdf5lib.exceptions.HDF5JavaException;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.zip.CRC32;
import org.apache.commons.io.FileUtils;

class ArchiveEntryExtractProcessor
implements IArchiveEntryProcessor {
    private static final int ROOT_UID = 0;
    private final IArchiveEntryVisitor visitorOrNull;
    private final ArchivingStrategy strategy;
    private final File rootDirectory;
    private final String rootPathToStrip;
    private final byte[] buffer;
    private final GroupCache groupCache;

    ArchiveEntryExtractProcessor(IArchiveEntryVisitor visitorOrNull, ArchivingStrategy strategy, File rootDirectory, String rootPathToStrip, byte[] buffer) {
        this.visitorOrNull = visitorOrNull;
        this.strategy = strategy;
        this.rootDirectory = rootDirectory;
        String normalizedRootPathToStrip = Utils.normalizePath(rootPathToStrip);
        this.rootPathToStrip = "/".equals(normalizedRootPathToStrip) ? "" : normalizedRootPathToStrip;
        this.buffer = buffer;
        this.groupCache = new GroupCache();
    }

    @Override
    public boolean process(String dir, String path, LinkRecord link, IHDF5Reader reader, IdCache idCache, IErrorStrategy errorStrategy) throws IOException {
        if (this.strategy.doExclude(path, link.isDirectory())) {
            return false;
        }
        File file2 = this.createFile(path);
        if (link.isDirectory()) {
            if (file2.exists() && !file2.isDirectory()) {
                file2.delete();
            }
            file2.mkdirs();
            if (!file2.isDirectory()) {
                errorStrategy.dealWithError(new UnarchivingException(file2, new IOException("Failed to make directory '" + file2.getAbsolutePath() + "'.")));
            }
            if (this.visitorOrNull != null) {
                this.visitorOrNull.visit(new ArchiveEntry(dir, path, link, idCache));
            }
        } else if (link.tryGetLinkTarget() != null && Unix.isOperational()) {
            try {
                file2.delete();
                String linkTarget = link.tryGetLinkTarget();
                Unix.createSymbolicLink(linkTarget, file2.getAbsolutePath());
                this.restoreAttributes(file2, link);
                if (this.visitorOrNull != null) {
                    this.visitorOrNull.visit(new ArchiveEntry(dir, path, link, idCache));
                }
            }
            catch (IOExceptionUnchecked ex) {
                errorStrategy.dealWithError(new UnarchivingException(file2, ex));
            }
        } else if (link.isSymLink()) {
            if (!Unix.isOperational()) {
                errorStrategy.warning("Warning: extracting symlink as regular file because Unix calls are not available on this system.");
            } else {
                errorStrategy.dealWithError(new UnarchivingException(path, new HDF5JavaException("Symlink doesn't have a link target.")));
            }
        } else {
            try {
                long size = reader.object().getSize(path);
                int crc32 = this.copyFromHDF5(reader, path, size, file2);
                this.restoreAttributes(file2, link);
                FileSizeType sizeType = this.getFileSizeType(file2);
                link.setVerifiedType(sizeType.type);
                link.setFileVerification(sizeType.size, crc32, file2.lastModified() / 1000L);
                ArchiveEntry entry = new ArchiveEntry(dir, path, link, idCache);
                if (this.visitorOrNull != null) {
                    this.visitorOrNull.visit(entry);
                }
                if (!entry.isOK()) {
                    errorStrategy.dealWithError(new UnarchivingException(path, entry.getStatus(true)));
                }
            }
            catch (IOException ex) {
                errorStrategy.dealWithError(new UnarchivingException(file2, ex));
            }
            catch (HDF5Exception ex) {
                errorStrategy.dealWithError(new UnarchivingException(path, ex));
            }
        }
        return true;
    }

    @Override
    public void postProcessDirectory(String dir, String path, LinkRecord link, IHDF5Reader reader, IdCache idCache, IErrorStrategy errorStrategy) throws IOException, HDF5Exception {
        File file2 = this.createFile(path);
        this.restoreAttributes(file2, link);
    }

    private File createFile(String path) {
        String workingPath = path.startsWith(this.rootPathToStrip) ? path.substring(this.rootPathToStrip.length()) : path;
        File file2 = new File(this.rootDirectory, workingPath);
        return file2;
    }

    private FileSizeType getFileSizeType(File file2) {
        if (Unix.isOperational()) {
            Unix.Stat info = Unix.getLinkInfo(file2.getPath(), false);
            return new FileSizeType(info.getLinkType(), info.getSize());
        }
        return new FileSizeType(file2.isDirectory() ? FileLinkType.DIRECTORY : (file2.isFile() ? FileLinkType.REGULAR_FILE : FileLinkType.OTHER), file2.length());
    }

    private int copyFromHDF5(IHDF5Reader reader, String objectPath, long size, File destination) throws IOException {
        Throwable throwable = null;
        Object var7_7 = null;
        try (FileOutputStream output = FileUtils.openOutputStream(destination);){
            CRC32 crc32 = new CRC32();
            long offset = 0L;
            while (offset < size) {
                int n = reader.opaque().readArrayToBlockWithOffset(objectPath, this.buffer, this.buffer.length, offset, 0);
                offset += (long)n;
                ((OutputStream)output).write(this.buffer, 0, n);
                crc32.update(this.buffer, 0, n);
            }
            ((OutputStream)output).close();
            return (int)crc32.getValue();
        }
        catch (Throwable throwable2) {
            if (throwable == null) {
                throwable = throwable2;
            } else if (throwable != throwable2) {
                throwable.addSuppressed(throwable2);
            }
            throw throwable;
        }
    }

    private void restoreAttributes(File file2, LinkRecord linkInfoOrNull) {
        assert (file2 != null);
        if (linkInfoOrNull != null) {
            if (linkInfoOrNull.hasLastModified()) {
                Unix.setLinkTimestamps(file2.getAbsolutePath(), linkInfoOrNull.getLastModified(), linkInfoOrNull.getLastModified());
            }
            if (linkInfoOrNull.hasUnixPermissions() && Unix.isOperational()) {
                if (!linkInfoOrNull.isSymLink()) {
                    Unix.setAccessMode(file2.getPath(), linkInfoOrNull.getPermissions());
                }
                if (Unix.getUid() == 0) {
                    Unix.setLinkOwner(file2.getPath(), linkInfoOrNull.getUid(), linkInfoOrNull.getGid());
                } else if (this.groupCache.isUserInGroup(linkInfoOrNull.getGid())) {
                    Unix.setLinkOwner(file2.getPath(), Unix.getUid(), linkInfoOrNull.getGid());
                }
            }
        }
    }

    @Override
    public ArchiverException createException(String objectPath, String detailedMsg) {
        return new UnarchivingException(objectPath, detailedMsg);
    }

    @Override
    public ArchiverException createException(String objectPath, HDF5Exception cause) {
        return new UnarchivingException(objectPath, cause);
    }

    @Override
    public ArchiverException createException(String objectPath, RuntimeException cause) {
        return new UnarchivingException(objectPath, cause);
    }

    @Override
    public ArchiverException createException(File file2, IOException cause) {
        return new UnarchivingException(file2, cause);
    }

    private static class FileSizeType {
        final FileLinkType type;
        final long size;

        FileSizeType(FileLinkType type, long size) {
            this.type = type;
            this.size = size;
        }
    }
}

