/*
 * Decompiled with CFR 0.152.
 */
package loci.formats.tiff;

import java.io.Closeable;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import loci.common.ByteArrayHandle;
import loci.common.DataTools;
import loci.common.RandomAccessInputStream;
import loci.common.Region;
import loci.common.enumeration.EnumException;
import loci.formats.FormatException;
import loci.formats.ImageTools;
import loci.formats.codec.CodecOptions;
import loci.formats.tiff.IFD;
import loci.formats.tiff.IFDList;
import loci.formats.tiff.IFDType;
import loci.formats.tiff.OnDemandLongArray;
import loci.formats.tiff.PhotoInterp;
import loci.formats.tiff.TiffCompression;
import loci.formats.tiff.TiffIFDEntry;
import loci.formats.tiff.TiffRational;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TiffParser
implements Closeable {
    private static final Logger LOGGER = LoggerFactory.getLogger(TiffParser.class);
    protected transient RandomAccessInputStream in;
    private byte[] cachedTileBuffer;
    private boolean bigTiff;
    private boolean fakeBigTiff = false;
    private boolean ycbcrCorrection = true;
    private boolean equalStrips = false;
    private boolean doCaching;
    private IFDList ifdList;
    private IFD firstIFD;
    private CodecOptions codecOptions = CodecOptions.getDefaultOptions();
    private boolean canClose = false;
    private String tagEncoding = null;

    public TiffParser(String filename) throws IOException {
        this(new RandomAccessInputStream(filename));
        this.canClose = true;
    }

    public TiffParser(RandomAccessInputStream in) {
        this.in = in;
        this.doCaching = true;
        try {
            long fp = in.getFilePointer();
            this.checkHeader();
            in.seek(fp);
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    @Override
    public void close() throws IOException {
        if (this.canClose && this.in != null) {
            this.in.close();
        }
    }

    public void setAssumeEqualStrips(boolean equalStrips) {
        this.equalStrips = equalStrips;
    }

    public void setCodecOptions(CodecOptions codecOptions) {
        this.codecOptions = codecOptions;
    }

    public CodecOptions getCodecOptions() {
        return this.codecOptions;
    }

    public void setDoCaching(boolean doCaching) {
        this.doCaching = doCaching;
    }

    public void setUse64BitOffsets(boolean use64Bit) {
        this.fakeBigTiff = use64Bit;
    }

    public void setYCbCrCorrection(boolean correctionAllowed) {
        this.ycbcrCorrection = correctionAllowed;
    }

    public void setEncoding(String encoding) {
        this.tagEncoding = encoding;
    }

    public RandomAccessInputStream getStream() {
        return this.in;
    }

    public boolean isValidHeader() {
        try {
            return this.checkHeader() != null;
        }
        catch (IOException e) {
            return false;
        }
    }

    public Boolean checkHeader() throws IOException {
        boolean bigEndian;
        if (this.in.length() < 4L) {
            return null;
        }
        this.in.seek(0L);
        int endianOne = this.in.read();
        int endianTwo = this.in.read();
        boolean littleEndian = endianOne == 73 && endianTwo == 73;
        boolean bl = bigEndian = endianOne == 77 && endianTwo == 77;
        if (!littleEndian && !bigEndian) {
            return null;
        }
        this.in.order(littleEndian);
        short magic = this.in.readShort();
        boolean bl2 = this.bigTiff = magic == 43;
        if (magic != 42 && magic != 43) {
            return null;
        }
        return littleEndian;
    }

    public boolean isBigTiff() {
        return this.bigTiff;
    }

    public IFDList getMainIFDs() throws IOException {
        if (this.ifdList != null) {
            return this.ifdList;
        }
        long[] offsets = this.getIFDOffsets();
        IFDList ifds = new IFDList();
        for (long offset : offsets) {
            IFD ifd = this.getIFD(offset);
            if (ifd == null || !ifd.containsKey(256)) continue;
            ifds.add(ifd);
        }
        if (this.doCaching) {
            this.ifdList = ifds;
        }
        return ifds;
    }

    public IFDList getSubIFDs(IFD ifd) throws IOException {
        IFDList list = new IFDList();
        long[] offsets = null;
        try {
            this.fillInIFD(ifd);
            offsets = ifd.getIFDLongArray(330);
        }
        catch (FormatException formatException) {
            // empty catch block
        }
        if (offsets != null) {
            for (long offset : offsets) {
                list.add(this.getIFD(offset));
            }
        }
        return list;
    }

    @Deprecated
    public IFDList getIFDs() throws IOException {
        if (this.ifdList != null) {
            return this.ifdList;
        }
        long[] offsets = this.getIFDOffsets();
        IFDList ifds = new IFDList();
        for (long offset : offsets) {
            IFD ifd = this.getIFD(offset);
            if (ifd == null) continue;
            if (ifd.containsKey(256)) {
                ifds.add(ifd);
            }
            long[] subOffsets = null;
            try {
                if (!this.doCaching && ifd.containsKey(330)) {
                    this.fillInIFD(ifd);
                }
                subOffsets = ifd.getIFDLongArray(330);
            }
            catch (FormatException formatException) {
                // empty catch block
            }
            if (subOffsets == null) continue;
            for (long subOffset : subOffsets) {
                IFD sub = this.getIFD(subOffset);
                if (sub == null) continue;
                ifds.add(sub);
            }
        }
        if (this.doCaching) {
            this.ifdList = ifds;
        }
        return ifds;
    }

    public IFDList getThumbnailIFDs() throws IOException {
        IFDList ifds = this.getMainIFDs();
        IFDList thumbnails = new IFDList();
        for (IFD ifd : ifds) {
            Number subfile = (Number)ifd.getIFDValue(254);
            int subfileType = subfile == null ? 0 : subfile.intValue();
            if (subfileType != 1) continue;
            thumbnails.add(ifd);
        }
        return thumbnails;
    }

    public IFDList getNonThumbnailIFDs() throws IOException {
        IFDList ifds = this.getMainIFDs();
        IFDList nonThumbs = new IFDList();
        for (IFD ifd : ifds) {
            int subfileType;
            Number subfile = (Number)ifd.getIFDValue(254);
            int n = subfileType = subfile == null ? 0 : subfile.intValue();
            if (subfileType == 1 && ifds.size() > 1) continue;
            nonThumbs.add(ifd);
        }
        return nonThumbs;
    }

    public IFDList getExifIFDs() throws FormatException, IOException {
        IFDList ifds = this.getMainIFDs();
        IFDList exif = new IFDList();
        for (IFD ifd : ifds) {
            IFD exifIFD;
            long offset = ifd.getIFDLongValue(34665, 0L);
            if (offset == 0L || (exifIFD = this.getIFD(offset)) == null) continue;
            exif.add(exifIFD);
        }
        return exif;
    }

    public long[] getIFDOffsets() throws IOException {
        int bytesPerEntry = this.bigTiff ? 20 : 12;
        ArrayList<Long> offsets = new ArrayList<Long>();
        long offset = this.getFirstOffset();
        while (offset > 0L && offset < this.in.length()) {
            this.in.seek(offset);
            offsets.add(offset);
            long nEntries = this.bigTiff ? this.in.readLong() : (long)this.in.readUnsignedShort();
            long entryBytes = nEntries * (long)bytesPerEntry;
            if (this.in.getFilePointer() + entryBytes + (long)(this.bigTiff ? 8 : 4) > this.in.length()) break;
            this.in.skipBytes(entryBytes);
            offset = this.getNextOffset(offset);
        }
        long[] f = new long[offsets.size()];
        for (int i = 0; i < f.length; ++i) {
            f[i] = (Long)offsets.get(i);
        }
        return f;
    }

    public IFD getFirstIFD() throws IOException {
        if (this.firstIFD != null) {
            return this.firstIFD;
        }
        long offset = this.getFirstOffset();
        IFD ifd = this.getIFD(offset);
        if (this.doCaching) {
            this.firstIFD = ifd;
        }
        return ifd;
    }

    public TiffIFDEntry getFirstIFDEntry(int tag) throws IOException {
        long offset = this.getFirstOffset();
        if (offset < 0L) {
            return null;
        }
        this.in.seek(offset);
        long numEntries = this.bigTiff ? this.in.readLong() : (long)this.in.readUnsignedShort();
        int i = 0;
        while ((long)i < numEntries) {
            this.in.seek(offset + (long)(this.bigTiff ? 8 : 2) + (long)((this.bigTiff ? 20 : 12) * i));
            TiffIFDEntry entry = this.readTiffIFDEntry();
            if (entry.getTag() == tag) {
                return entry;
            }
            ++i;
        }
        throw new IllegalArgumentException("Unknown tag: " + tag);
    }

    public long getFirstOffset() throws IOException {
        Boolean header = this.checkHeader();
        if (header == null) {
            return -1L;
        }
        if (this.bigTiff) {
            this.in.skipBytes(4);
        }
        return this.getNextOffset(0L);
    }

    public IFD getIFD(long offset) throws IOException {
        long newOffset;
        if (offset < 0L || offset >= this.in.length()) {
            return null;
        }
        IFD ifd = new IFD();
        ifd.put(0, this.in.isLittleEndian());
        ifd.put(1, this.bigTiff);
        LOGGER.trace("getIFD: seeking IFD at {}", (Object)offset);
        this.in.seek(offset);
        long numEntries = this.bigTiff ? this.in.readLong() : (long)this.in.readUnsignedShort();
        LOGGER.trace("getIFD: {} directory entries to read", (Object)numEntries);
        if (numEntries == 0L || numEntries == 1L) {
            return ifd;
        }
        int bytesPerEntry = this.bigTiff ? 20 : 12;
        int baseOffset = this.bigTiff ? 8 : 2;
        int i = 0;
        while ((long)i < numEntries) {
            this.in.seek(offset + (long)baseOffset + (long)(bytesPerEntry * i));
            TiffIFDEntry entry = null;
            try {
                entry = this.readTiffIFDEntry();
            }
            catch (EnumException e) {
                LOGGER.debug("", (Throwable)e);
            }
            if (entry == null) break;
            int count = entry.getValueCount();
            int tag = entry.getTag();
            long pointer = entry.getValueOffset();
            int bpe = entry.getType().getBytesPerElement();
            if (count < 0 || bpe <= 0) {
                this.in.skipBytes(bytesPerEntry - 4 - (this.bigTiff ? 8 : 4));
            } else {
                Object value = null;
                long inputLen = this.in.length();
                if ((long)(count * bpe) + pointer > inputLen) {
                    int oldCount = count;
                    count = (int)((inputLen - pointer) / (long)bpe);
                    LOGGER.trace("getIFD: truncated {} array elements for tag {}", (Object)(oldCount - count), (Object)tag);
                    if (count < 0) {
                        count = oldCount;
                    }
                }
                if (count < 0 || (long)count > this.in.length()) break;
                value = pointer != this.in.getFilePointer() && !this.doCaching ? entry : this.getIFDValue(entry);
                if (value != null && !ifd.containsKey(tag)) {
                    ifd.put(tag, value);
                }
            }
            ++i;
        }
        if ((newOffset = offset + (long)baseOffset + (long)bytesPerEntry * numEntries) < this.in.length()) {
            this.in.seek(newOffset);
        } else {
            this.in.seek(this.in.length());
        }
        return ifd;
    }

    public void fillInIFD(IFD ifd) throws IOException {
        HashSet<TiffIFDEntry> entries = new HashSet<TiffIFDEntry>();
        for (Object key : ifd.keySet()) {
            if (!(ifd.get(key) instanceof TiffIFDEntry)) continue;
            entries.add((TiffIFDEntry)ifd.get(key));
        }
        for (TiffIFDEntry entry : entries) {
            if (entry.getValueCount() >= 0xA00000 && entry.getTag() >= 32768 || entry.getTag() == 320) continue;
            ifd.put(entry.getTag(), this.getIFDValue(entry));
        }
    }

    public Object getIFDValue(TiffIFDEntry entry) throws IOException {
        IFDType type = entry.getType();
        int count = entry.getValueCount();
        long offset = entry.getValueOffset();
        LOGGER.trace("Reading entry {} from {}; type={}, count={}", new Object[]{entry.getTag(), offset, type, count});
        if (offset >= this.in.length()) {
            return null;
        }
        if (offset != this.in.getFilePointer()) {
            if (this.fakeBigTiff && offset < 0L) {
                offset &= 0xFFFFFFFFL;
                offset += 0x100000000L;
            }
            if (offset >= this.in.length()) {
                return null;
            }
            this.in.seek(offset);
        }
        if (type == IFDType.BYTE) {
            if (count == 1) {
                return this.in.readUnsignedByte();
            }
            byte[] bytes = new byte[count];
            this.in.readFully(bytes);
            short[] shorts = new short[count];
            for (int j = 0; j < count; ++j) {
                shorts[j] = (short)(bytes[j] & 0xFF);
            }
            return shorts;
        }
        if (type == IFDType.ASCII) {
            byte[] ascii = new byte[count];
            this.in.read(ascii);
            int nullCount = 0;
            for (int j = 0; j < count; ++j) {
                if (ascii[j] != 0 && j != count - 1) continue;
                ++nullCount;
            }
            String[] strings = nullCount == 1 ? null : new String[nullCount];
            Object s = null;
            int c = 0;
            int ndx = -1;
            String encoding = this.tagEncoding == null ? "UTF-8" : this.tagEncoding;
            for (int j = 0; j < count; ++j) {
                if (ascii[j] == 0) {
                    s = new String(ascii, ndx + 1, j - ndx - 1, encoding);
                    ndx = j;
                } else {
                    s = j == count - 1 ? new String(ascii, ndx + 1, j - ndx, encoding) : null;
                }
                if (strings == null || s == null) continue;
                strings[c++] = s;
            }
            return strings == null ? s : strings;
        }
        if (type == IFDType.SHORT) {
            if (count == 1) {
                return this.in.readUnsignedShort();
            }
            int[] shorts = new int[count];
            for (int j = 0; j < count; ++j) {
                shorts[j] = this.in.readUnsignedShort();
            }
            return shorts;
        }
        if (type == IFDType.LONG || type == IFDType.IFD) {
            if (count == 1) {
                return this.in.readUnsignedInt();
            }
            long[] longs = new long[count];
            for (int j = 0; j < count; ++j) {
                if (this.in.getFilePointer() + 4L > this.in.length()) continue;
                longs[j] = this.in.readUnsignedInt();
            }
            return longs;
        }
        if (type == IFDType.LONG8 || type == IFDType.SLONG8 || type == IFDType.IFD8) {
            if (count == 1) {
                return this.in.readLong();
            }
            long[] longs = null;
            if (this.equalStrips && (entry.getTag() == 279 || entry.getTag() == 325)) {
                longs = new long[]{this.in.readLong()};
            } else {
                if (entry.getTag() == 273 || entry.getTag() == 324 || entry.getTag() == 279 || entry.getTag() == 325) {
                    OnDemandLongArray offsets = new OnDemandLongArray(this.in);
                    offsets.setSize(count);
                    return offsets;
                }
                longs = new long[count];
                for (int j = 0; j < count; ++j) {
                    longs[j] = this.in.readLong();
                }
            }
            return longs;
        }
        if (type == IFDType.RATIONAL || type == IFDType.SRATIONAL) {
            if (count == 1) {
                return new TiffRational(this.in.readUnsignedInt(), this.in.readUnsignedInt());
            }
            TiffRational[] rationals = new TiffRational[count];
            for (int j = 0; j < count; ++j) {
                rationals[j] = new TiffRational(this.in.readUnsignedInt(), this.in.readUnsignedInt());
            }
            return rationals;
        }
        if (type == IFDType.SBYTE || type == IFDType.UNDEFINED) {
            if (count == 1) {
                return this.in.readByte();
            }
            byte[] sbytes = new byte[count];
            this.in.read(sbytes);
            return sbytes;
        }
        if (type == IFDType.SSHORT) {
            if (count == 1) {
                return this.in.readShort();
            }
            short[] sshorts = new short[count];
            for (int j = 0; j < count; ++j) {
                sshorts[j] = this.in.readShort();
            }
            return sshorts;
        }
        if (type == IFDType.SLONG) {
            if (count == 1) {
                return this.in.readInt();
            }
            int[] slongs = new int[count];
            for (int j = 0; j < count; ++j) {
                slongs[j] = this.in.readInt();
            }
            return slongs;
        }
        if (type == IFDType.FLOAT) {
            if (count == 1) {
                return Float.valueOf(this.in.readFloat());
            }
            float[] floats = new float[count];
            for (int j = 0; j < count; ++j) {
                floats[j] = this.in.readFloat();
            }
            return floats;
        }
        if (type == IFDType.DOUBLE) {
            if (count == 1) {
                return this.in.readDouble();
            }
            double[] doubles = new double[count];
            for (int j = 0; j < count; ++j) {
                doubles[j] = this.in.readDouble();
            }
            return doubles;
        }
        return null;
    }

    public String getComment() throws IOException {
        IFD firstIFD = this.getFirstIFD();
        if (firstIFD == null) {
            return null;
        }
        this.fillInIFD(firstIFD);
        return firstIFD.getComment();
    }

    public int[] getColorMap(IFD ifd) throws IOException {
        Object map = ifd.get(320);
        if (map == null) {
            return null;
        }
        int[] colorMap = null;
        if (map instanceof TiffIFDEntry) {
            colorMap = (int[])this.getIFDValue((TiffIFDEntry)map);
        } else if (map instanceof int[]) {
            colorMap = (int[])map;
        }
        return colorMap;
    }

    public byte[] getTile(IFD ifd, byte[] buf, int row, int col) throws FormatException, IOException {
        int realBytes;
        int channel;
        Object stripOffsets;
        int offsetIndex;
        OnDemandLongArray counts;
        int effectiveChannels;
        byte[] jpegTable = (byte[])ifd.getIFDValue(347);
        this.codecOptions.interleaved = true;
        this.codecOptions.littleEndian = ifd.isLittleEndian();
        long tileWidth = ifd.getTileWidth();
        long tileLength = ifd.getTileLength();
        int samplesPerPixel = ifd.getSamplesPerPixel();
        int planarConfig = ifd.getPlanarConfiguration();
        TiffCompression compression = ifd.getCompression();
        long numTileCols = ifd.getTilesPerRow();
        int pixel = ifd.getBytesPerSample()[0];
        int n = effectiveChannels = planarConfig == 2 ? 1 : samplesPerPixel;
        if (ifd.get(279) instanceof OnDemandLongArray && (counts = (OnDemandLongArray)ifd.get(279)) != null) {
            counts.setStream(this.in);
        }
        if (ifd.get(325) instanceof OnDemandLongArray && (counts = (OnDemandLongArray)ifd.get(325)) != null) {
            counts.setStream(this.in);
        }
        long[] stripByteCounts = ifd.getStripByteCounts();
        long[] rowsPerStrip = ifd.getRowsPerStrip();
        int countIndex = offsetIndex = (int)((long)row * numTileCols + (long)col);
        if (this.equalStrips) {
            countIndex = 0;
        }
        if (stripByteCounts[countIndex] == rowsPerStrip[0] * tileWidth && pixel > 1) {
            int n2 = countIndex;
            stripByteCounts[n2] = stripByteCounts[n2] * (long)pixel;
        } else if (stripByteCounts[countIndex] < 0L && countIndex > 0) {
            LOGGER.debug("byte count #{} was {}; correcting to {}", new Object[]{countIndex, stripByteCounts[countIndex], stripByteCounts[countIndex - 1]});
            stripByteCounts[countIndex] = stripByteCounts[countIndex - 1];
        }
        long stripOffset = 0L;
        long nStrips = 0L;
        if (ifd.getOnDemandStripOffsets() != null) {
            stripOffsets = ifd.getOnDemandStripOffsets();
            ((OnDemandLongArray)stripOffsets).setStream(this.in);
            stripOffset = ((OnDemandLongArray)stripOffsets).get(offsetIndex);
            nStrips = ((OnDemandLongArray)stripOffsets).size();
        } else {
            stripOffsets = ifd.getStripOffsets();
            stripOffset = (long)stripOffsets[offsetIndex];
            nStrips = ((Object)stripOffsets).length;
        }
        int size = (int)(tileWidth * tileLength * (long)pixel * (long)effectiveChannels);
        if (buf == null) {
            buf = new byte[size];
        }
        if (stripByteCounts[countIndex] == 0L || stripOffset >= this.in.length()) {
            Arrays.fill(buf, (byte)0);
            return buf;
        }
        int tileSize = (int)stripByteCounts[countIndex];
        if (jpegTable != null) {
            tileSize += jpegTable.length - 2;
        }
        byte[] tile = new byte[tileSize];
        LOGGER.debug("Reading tile Length {} Offset {}", (Object)tile.length, (Object)stripOffset);
        if (jpegTable != null) {
            System.arraycopy(jpegTable, 0, tile, 0, jpegTable.length - 2);
            this.in.seek(stripOffset + 2L);
            this.in.read(tile, jpegTable.length - 2, tile.length - (jpegTable.length - 2));
        } else {
            this.in.seek(stripOffset);
            this.in.read(tile);
        }
        if (ifd.getIFDIntValue(266) == 2 && (compression.getCode() <= TiffCompression.GROUP_4_FAX.getCode() || compression.getCode() == TiffCompression.DEFLATE.getCode() || compression.getCode() == TiffCompression.PROPRIETARY_DEFLATE.getCode())) {
            for (int i = 0; i < tile.length; ++i) {
                tile[i] = (byte)(Integer.reverse(tile[i]) >> 24);
            }
        }
        this.codecOptions.maxBytes = Math.max(size, tile.length);
        this.codecOptions.ycbcr = ifd.getPhotometricInterpretation() == PhotoInterp.Y_CB_CR && ifd.getIFDIntValue(530) == 1 && this.ycbcrCorrection;
        tile = compression.decompress(tile, this.codecOptions);
        TiffCompression.undifference(tile, ifd);
        TiffParser.unpackBytes(buf, 0, tile, ifd);
        if (planarConfig == 2 && !ifd.isTiled() && ifd.getSamplesPerPixel() > 1 && (channel = (int)((long)row % nStrips)) < ifd.getBytesPerSample().length && (realBytes = ifd.getBytesPerSample()[channel]) != pixel) {
            int i;
            boolean littleEndian = ifd.isLittleEndian();
            int[] samples = new int[buf.length / pixel];
            for (i = 0; i < samples.length; ++i) {
                samples[i] = DataTools.bytesToInt(buf, i * realBytes, realBytes, littleEndian);
            }
            for (i = 0; i < samples.length; ++i) {
                DataTools.unpackBytes(samples[i], buf, i * pixel, pixel, littleEndian);
            }
        }
        return buf;
    }

    public byte[] getSamples(IFD ifd, byte[] buf) throws FormatException, IOException {
        long width = ifd.getImageWidth();
        long length = ifd.getImageLength();
        return this.getSamples(ifd, buf, 0, 0, width, length);
    }

    public byte[] getSamples(IFD ifd, byte[] buf, int x, int y, long width, long height) throws FormatException, IOException {
        return this.getSamples(ifd, buf, x, y, width, height, 0, 0);
    }

    public byte[] getSamples(IFD ifd, byte[] buf, int x, int y, long width, long height, int overlapX, int overlapY) throws FormatException, IOException {
        boolean contiguousTiles;
        OnDemandLongArray counts;
        int effectiveChannels;
        LOGGER.trace("parsing IFD entries");
        boolean littleEndian = ifd.isLittleEndian();
        this.in.order(littleEndian);
        int samplesPerPixel = ifd.getSamplesPerPixel();
        long tileWidth = ifd.getTileWidth();
        long tileLength = ifd.getTileLength();
        if (tileLength <= 0L) {
            LOGGER.trace("Tile length is {}; setting it to {}", (Object)tileLength, (Object)height);
            tileLength = height;
        }
        long numTileRows = ifd.getTilesPerColumn();
        long numTileCols = ifd.getTilesPerRow();
        PhotoInterp photoInterp = ifd.getPhotometricInterpretation();
        int planarConfig = ifd.getPlanarConfiguration();
        int pixel = ifd.getBytesPerSample()[0];
        int n = effectiveChannels = planarConfig == 2 ? 1 : samplesPerPixel;
        if (LOGGER.isTraceEnabled()) {
            ifd.printIFD();
        }
        if (width * height > Integer.MAX_VALUE) {
            throw new FormatException("Sorry, ImageWidth x ImageLength > 2147483647 is not supported (" + width + " x " + height + ")");
        }
        if (width * height * (long)effectiveChannels * (long)pixel > Integer.MAX_VALUE) {
            throw new FormatException("Sorry, ImageWidth x ImageLength x SamplesPerPixel x BitsPerSample > 2147483647 is not supported (" + width + " x " + height + " x " + samplesPerPixel + " x " + pixel * 8 + ")");
        }
        int numSamples = (int)(width * height);
        LOGGER.trace("reading image data (samplesPerPixel={}; numSamples={})", (Object)samplesPerPixel, (Object)numSamples);
        TiffCompression compression = ifd.getCompression();
        this.codecOptions = compression == TiffCompression.JPEG_2000 || compression == TiffCompression.JPEG_2000_LOSSY ? compression.getCompressionCodecOptions(ifd, this.codecOptions) : compression.getCompressionCodecOptions(ifd);
        this.codecOptions.interleaved = true;
        this.codecOptions.littleEndian = ifd.isLittleEndian();
        long imageWidth = ifd.getImageWidth();
        long imageLength = ifd.getImageLength();
        long[] stripOffsets = null;
        if (ifd.getOnDemandStripOffsets() != null) {
            OnDemandLongArray offsets = ifd.getOnDemandStripOffsets();
            offsets.setStream(this.in);
            stripOffsets = offsets.toArray();
        } else {
            stripOffsets = ifd.getStripOffsets();
        }
        if (ifd.get(279) instanceof OnDemandLongArray && (counts = (OnDemandLongArray)ifd.get(279)) != null) {
            counts.setStream(this.in);
        }
        if (ifd.get(325) instanceof OnDemandLongArray && (counts = (OnDemandLongArray)ifd.get(325)) != null) {
            counts.setStream(this.in);
        }
        long[] stripByteCounts = ifd.getStripByteCounts();
        boolean bl = contiguousTiles = tileWidth == imageWidth && planarConfig == 1;
        if (contiguousTiles) {
            for (int i = 1; i < stripOffsets.length; ++i) {
                if (stripOffsets[i] == stripOffsets[i - 1] + stripByteCounts[i - 1] && stripOffsets[i] + stripByteCounts[i] <= this.in.length()) continue;
                contiguousTiles = false;
                break;
            }
        }
        if (!(effectiveChannels != 1 && planarConfig != 1 || ifd.getBitsPerSample()[0] % 8 != 0 || photoInterp == PhotoInterp.WHITE_IS_ZERO || photoInterp == PhotoInterp.CMYK || photoInterp == PhotoInterp.Y_CB_CR || compression != TiffCompression.UNCOMPRESSED || ifd.getIFDIntValue(266) == 2 || stripOffsets == null || stripByteCounts == null || this.in.length() < stripOffsets[0] + stripByteCounts[0] || numTileRows * numTileCols != 1L && !contiguousTiles)) {
            if (contiguousTiles) {
                stripByteCounts = new long[]{stripByteCounts[0] * (long)stripByteCounts.length};
                stripOffsets = new long[]{stripOffsets[0]};
                tileLength = imageLength;
            }
            long column = (long)x / tileWidth;
            int firstTile = (int)((long)y / tileLength * numTileCols + column);
            int lastTile = (int)(((long)y + height) / tileLength * numTileCols + column);
            lastTile = Math.min(lastTile, stripOffsets.length - 1);
            if (planarConfig == 2) {
                lastTile = stripOffsets.length - 1;
            }
            int bytes = ifd.getBitsPerSample()[0] / 8;
            int offset = 0;
            block1: for (int tile = firstTile; tile <= lastTile; ++tile) {
                long byteCount;
                long l = byteCount = this.equalStrips ? stripByteCounts[0] : stripByteCounts[tile];
                if (byteCount == (long)numSamples && pixel > 1) {
                    byteCount *= (long)pixel;
                }
                if (stripOffsets[tile] >= this.in.length()) continue;
                this.in.seek(stripOffsets[tile]);
                if (width == tileWidth && height == imageLength) {
                    int len = (int)Math.min((long)(buf.length - offset), byteCount);
                    this.in.read(buf, offset, len);
                    offset += len;
                    continue;
                }
                int bpp = bytes * effectiveChannels;
                this.in.skipBytes((long)(y * bpp) * tileWidth);
                int row = 0;
                while ((long)row < height) {
                    this.in.skipBytes(x * bpp);
                    int len = (int)Math.min((long)(buf.length - offset), width * (long)bpp);
                    if (len <= 0) continue block1;
                    this.in.read(buf, offset, len);
                    offset += len;
                    int skip = (int)((long)bpp * (tileWidth - (long)x - width));
                    if ((long)skip + this.in.getFilePointer() < this.in.length()) {
                        this.in.skipBytes(skip);
                    }
                    ++row;
                }
            }
            if (effectiveChannels > 1) {
                int c;
                byte[][] split = new byte[effectiveChannels][buf.length / effectiveChannels];
                for (c = 0; c < split.length; ++c) {
                    split[c] = ImageTools.splitChannels(buf, c, effectiveChannels, bytes, false, true);
                }
                for (c = 0; c < split.length; ++c) {
                    System.arraycopy(split[c], 0, buf, c * split[c].length, split[c].length);
                }
            }
            return buf;
        }
        if (numTileRows > Integer.MAX_VALUE) {
            throw new FormatException(numTileRows + " rows of tiles not supported");
        }
        int nrows = (int)numTileRows;
        if (planarConfig == 2) {
            numTileRows *= (long)samplesPerPixel;
        }
        Region imageBounds = new Region(x, y, (int)width, (int)height);
        int endX = (int)width + x;
        int endY = (int)height + y;
        long w = tileWidth;
        long h = tileLength;
        int rowLen = pixel * (int)w;
        int tileSize = (int)((long)rowLen * h);
        int planeSize = (int)(width * height * (long)pixel);
        int outputRowLen = (int)((long)pixel * width);
        int bufferSizeSamplesPerPixel = samplesPerPixel;
        if (ifd.getPlanarConfiguration() == 2) {
            bufferSizeSamplesPerPixel = 1;
        }
        int bpp = ifd.getBytesPerSample()[0];
        int bufferSize = (int)tileWidth * (int)tileLength * bufferSizeSamplesPerPixel * bpp;
        this.cachedTileBuffer = new byte[bufferSize];
        Region tileBounds = new Region(0, 0, (int)tileWidth, (int)tileLength);
        int row = 0;
        while ((long)row < numTileRows) {
            if (row == 0) {
                tileBounds.height = (int)(tileLength - (long)overlapY);
            }
            int col = 0;
            while ((long)col < numTileCols) {
                if (col == 0) {
                    tileBounds.width = (int)(tileWidth - (long)overlapX);
                }
                tileBounds.x = col * (int)(tileWidth - (long)overlapX);
                tileBounds.y = row * (int)(tileLength - (long)overlapY);
                if (planarConfig == 2) {
                    tileBounds.y = (int)((long)(row % nrows) * (tileLength - (long)overlapY));
                }
                if (tileBounds.x > imageBounds.x + imageBounds.width) break;
                if (imageBounds.intersects(tileBounds)) {
                    int theight;
                    this.getTile(ifd, this.cachedTileBuffer, row, col);
                    int tileX = Math.max(tileBounds.x, x);
                    int tileY = Math.max(tileBounds.y, y);
                    int realX = tileX % (int)(tileWidth - (long)overlapX);
                    int realY = tileY % (int)(tileLength - (long)overlapY);
                    int twidth = (int)Math.min((long)(endX - tileX), tileWidth - (long)realX);
                    if (twidth <= 0) {
                        twidth = (int)Math.max((long)(endX - tileX), tileWidth - (long)realX);
                    }
                    if ((theight = (int)Math.min((long)(endY - tileY), tileLength - (long)realY)) <= 0) {
                        theight = (int)Math.max((long)(endY - tileY), tileLength - (long)realY);
                    }
                    int copy = pixel * twidth;
                    realX *= pixel;
                    realY *= rowLen;
                    for (int q = 0; q < effectiveChannels; ++q) {
                        int src = q * tileSize + realX + realY;
                        int dest = q * planeSize + pixel * (tileX - x) + outputRowLen * (tileY - y);
                        if (planarConfig == 2) {
                            dest += planeSize * (row / nrows);
                        }
                        if (rowLen == outputRowLen && overlapX == 0 && overlapY == 0 && rowLen == pixel * imageBounds.intersection((Region)tileBounds).width) {
                            System.arraycopy(this.cachedTileBuffer, src, buf, dest, copy * theight);
                            continue;
                        }
                        for (int tileRow = 0; tileRow < theight; ++tileRow) {
                            System.arraycopy(this.cachedTileBuffer, src, buf, dest, copy);
                            src += rowLen;
                            dest += outputRowLen;
                        }
                    }
                }
                ++col;
            }
            ++row;
        }
        return buf;
    }

    public static void unpackBytes(byte[] samples, int startIndex, byte[] bytes, IFD ifd) throws FormatException {
        int[] reference;
        float lumaBlue;
        float lumaGreen;
        float lumaRed;
        int skipBits;
        long maxValue;
        boolean littleEndian;
        boolean noDiv8;
        int nSamples;
        int numBytes;
        long imageWidth;
        int bps0;
        int sampleCount;
        int nChannels;
        PhotoInterp photoInterp;
        block35: {
            boolean bps16;
            boolean planar = ifd.getPlanarConfiguration() == 2;
            TiffCompression compression = ifd.getCompression();
            photoInterp = ifd.getPhotometricInterpretation();
            if (compression == TiffCompression.JPEG || compression == TiffCompression.JPEGXR) {
                photoInterp = PhotoInterp.RGB;
            }
            int[] bitsPerSample = ifd.getBitsPerSample();
            nChannels = bitsPerSample.length;
            sampleCount = (int)(8L * (long)bytes.length / (long)bitsPerSample[0]);
            if (photoInterp == PhotoInterp.Y_CB_CR) {
                sampleCount *= 3;
            }
            if (planar) {
                nChannels = 1;
            } else {
                sampleCount /= nChannels;
            }
            if (LOGGER.isTraceEnabled()) {
                LOGGER.trace("unpacking {} samples (startIndex={}; totalBits={}; numBytes={})", new Object[]{sampleCount, startIndex, nChannels * bitsPerSample[0], bytes.length});
            }
            boolean bps8 = (bps0 = bitsPerSample[0]) == 8;
            boolean bl = bps16 = bps0 == 16;
            if ((bps8 || bps16) && bytes.length <= samples.length && nChannels == 1 && photoInterp != PhotoInterp.WHITE_IS_ZERO && photoInterp != PhotoInterp.CMYK && photoInterp != PhotoInterp.Y_CB_CR) {
                System.arraycopy(bytes, 0, samples, 0, bytes.length);
                return;
            }
            imageWidth = ifd.getImageWidth();
            long imageHeight = ifd.getImageLength();
            numBytes = ifd.getBytesPerSample()[0];
            nSamples = samples.length / (nChannels * numBytes);
            noDiv8 = bps0 % 8 != 0;
            littleEndian = ifd.isLittleEndian();
            maxValue = (long)Math.pow(2.0, bps0) - 1L;
            if (photoInterp == PhotoInterp.CMYK) {
                maxValue = Integer.MAX_VALUE;
            }
            if ((skipBits = (int)(8L - imageWidth * (long)bps0 * (long)nChannels % 8L)) == 8 || (long)(bytes.length * 8) < (long)bps0 * ((long)nChannels * imageWidth + imageHeight)) {
                skipBits = 0;
            }
            lumaRed = 0.299f;
            lumaGreen = 0.587f;
            lumaBlue = 0.114f;
            reference = new int[]{0, 0, 0, 0, 0, 0};
            try {
                int[] value = ifd.getIFDIntArray(532);
                if (value == null) break block35;
                reference = value;
            }
            catch (FormatException e) {
                float[] value = (float[])ifd.getIFDValue(532);
                if (value == null || value.length != 6) break block35;
                LOGGER.debug("ReferenceBlackWhite tag stored as float array.");
                for (int i = 0; i < 5; ++i) {
                    reference[i] = (int)value[i];
                }
            }
        }
        int[] subsampling = ifd.getIFDIntArray(530);
        TiffRational[] coefficients = (TiffRational[])ifd.getIFDValue(529);
        if (coefficients != null) {
            lumaRed = coefficients[0].floatValue();
            lumaGreen = coefficients[1].floatValue();
            lumaBlue = coefficients[2].floatValue();
        }
        int subX = subsampling == null ? 2 : subsampling[0];
        int subY = subsampling == null ? 2 : subsampling[1];
        int block = subX * subY;
        int nTiles = (int)(imageWidth / (long)subX);
        RandomAccessInputStream bb = null;
        try {
            if (noDiv8) {
                bb = new RandomAccessInputStream(new ByteArrayHandle(bytes));
            }
            block14: for (int sample = 0; sample < sampleCount; ++sample) {
                int ndx = startIndex + sample;
                if (ndx >= nSamples) {
                    break;
                }
                for (int channel = 0; channel < nChannels; ++channel) {
                    int index = numBytes * (sample * nChannels + channel);
                    int outputIndex = (channel * nSamples + ndx) * numBytes;
                    if (photoInterp != PhotoInterp.Y_CB_CR) {
                        long value = 0L;
                        if (noDiv8) {
                            if (channel == 0 && photoInterp == PhotoInterp.RGB_PALETTE || photoInterp != PhotoInterp.CFA_ARRAY && photoInterp != PhotoInterp.RGB_PALETTE) {
                                try {
                                    value = bb.readBits(bps0) & 0xFFFF;
                                }
                                catch (ArrayIndexOutOfBoundsException arrayIndexOutOfBoundsException) {
                                    // empty catch block
                                }
                                if ((long)ndx % imageWidth == imageWidth - 1L) {
                                    bb.skipBits(skipBits);
                                }
                            }
                        } else {
                            value = numBytes == 1 ? (long)(bytes[index] & 0xFF) : DataTools.bytesToLong(bytes, index, numBytes, littleEndian);
                        }
                        if (photoInterp == PhotoInterp.WHITE_IS_ZERO || photoInterp == PhotoInterp.CMYK) {
                            value = maxValue - value;
                        }
                        if (outputIndex + numBytes > samples.length) continue;
                        if (numBytes == 1) {
                            samples[outputIndex] = (byte)value;
                            continue;
                        }
                        DataTools.unpackBytes(value, samples, outputIndex, numBytes, littleEndian);
                        continue;
                    }
                    if (channel != nChannels - 1) continue;
                    int lumaIndex = sample + 2 * (sample / block);
                    int chromaIndex = sample / block * (block + 2) + block;
                    if (chromaIndex + 1 >= bytes.length) continue block14;
                    int tile = ndx / block;
                    int pixel = ndx % block;
                    long r = subY * (tile / nTiles) + pixel / subX;
                    long c = subX * (tile % nTiles) + pixel % subX;
                    int idx = (int)(r * imageWidth + c);
                    if (idx >= nSamples) continue;
                    int y = (bytes[lumaIndex] & 0xFF) - reference[0];
                    int cb = (bytes[chromaIndex] & 0xFF) - reference[2];
                    int cr = (bytes[chromaIndex + 1] & 0xFF) - reference[4];
                    int red = (int)((float)cr * (2.0f - 2.0f * lumaRed) + (float)y);
                    int blue = (int)((float)cb * (2.0f - 2.0f * lumaBlue) + (float)y);
                    int green = (int)(((float)y - lumaBlue * (float)blue - lumaRed * (float)red) / lumaGreen);
                    samples[idx] = (byte)(red & 0xFF);
                    samples[nSamples + idx] = (byte)(green & 0xFF);
                    samples[2 * nSamples + idx] = (byte)(blue & 0xFF);
                }
            }
        }
        catch (IOException e) {
            throw new FormatException((Throwable)e);
        }
        finally {
            if (bb != null) {
                try {
                    bb.close();
                }
                catch (IOException e) {
                    throw new FormatException((Throwable)e);
                }
            }
        }
    }

    long getNextOffset(long previous) throws IOException {
        if (this.bigTiff || this.fakeBigTiff) {
            return this.in.readLong();
        }
        long offset = previous & 0xFFFFFFFF00000000L | this.in.readUnsignedInt();
        if (offset < previous && offset != 0L && (double)this.in.length() >= Math.pow(2.0, 32.0)) {
            offset += 0x100000000L;
        }
        return offset;
    }

    TiffIFDEntry readTiffIFDEntry() throws IOException {
        int valueCount;
        IFDType entryType;
        int entryTag = this.in.readUnsignedShort();
        try {
            entryType = IFDType.get(this.in.readUnsignedShort());
        }
        catch (EnumException e) {
            LOGGER.error("Error reading IFD type at: {}", (Object)this.in.getFilePointer());
            throw e;
        }
        int n = valueCount = this.bigTiff ? (int)this.in.readLong() : this.in.readInt();
        if (valueCount < 0) {
            throw new RuntimeException("Count of '" + valueCount + "' unexpected.");
        }
        int nValueBytes = valueCount * entryType.getBytesPerElement();
        int threshhold = this.bigTiff ? 8 : 4;
        long offset = nValueBytes > threshhold ? this.getNextOffset(0L) : this.in.getFilePointer();
        return new TiffIFDEntry(entryTag, entryType, valueCount, offset);
    }
}

